// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "degub.h"
#include "cpu.h"
#include "opcode.h"
#include "ini.h"
#include "stringCommons.h"
#include <math.h>

DWORD g_C87RC[4] = { _RC_NEAR, _RC_CHOP, _RC_UP, _RC_DOWN };

//ps and fps instructions assumes that HID2_PSE is always enabled
//fpd assumes it's disabled
//I think... :}

#define FPR_PS_D(fr) (r.fpr[fr].d)
#define GET_FPR_SINGLE(fr) FPR_PS_D(fr)
#define SET_FPRD(_d) { r.fpr[frD].d = (_d); }
#define SET_FPRPS(_ps0, _ps1) { PS0(frD) = (_ps0); PS1(frD) = (_ps1); }
#define SET_FPR_SINGLE(f) { double sfs = (f); SET_FPRPS(sfs, sfs); }
#define SET_FPR_SINGLE_FPRF(f) { double sfsf = (f); SET_FPR_SINGLE(sfsf);\
	r.set_fpscr_fprf(sfsf); }
#define SET_FPRD_FPRF(f) { double sdf = (f); SET_FPRD(sdf);\
	r.set_fpscr_fprf(sdf); }
#define SET_FPRPS_FPRF(_ps0, _ps1) { r.set_fpscr_fprf(PS0(frD) = (_ps0));\
	PS1(frD) = (_ps1); }

Asms g_asms;

Asms::Asms() {
	for(DWORD i=0; i<32; i++) {
		if(i == 0x14)
			BOasm[i] = "a";
		else if((getbitr(i, 1) && getbitr(i, 2)) ||
			(getbitr(i, 3) && getbitr(i, 4)) || i == 0x15)
			BOasm[i] = "UNDEF";
		else {
			BOasm[i] = getbitr(i, 0) ? "h" : "";
			if(!getbitr(i, 2))
				BOasm[i] = BOasm[i] + "d" + (getbitr(i, 1) ? "" : "n") + "z";
			if(!getbitr(i, 4))
				BOasm[i] = BOasm[i] + (getbitr(i, 3) ? "t" : "f");
		}

#define CASE_CPY(i, src) case i: BIasm[i] = src; break;
		switch(i % 4) {
			CASE_CPY(0, "LT");
			CASE_CPY(1, "GT");
			CASE_CPY(2, "EQ");
			CASE_CPY(3, "SO");
		}
		BIasm[i] += ('0' + char(i / 4));
	}
}

void CPU::__undefined() {
	ostringstream str;
	str << "Undefined opcode detected at " << HEX0x08(cia) << ": " << HEX08(opcode);
	g_errorstring = str.str();
	setQuit();
	error = true;
}

//----------------------------------OPCODES-------------------------------------

void CPU::_stw() {
	D_rS; D_rA; D_d16;
	m.ww(rA_0 + d, r.gpr[rS]);
}

void CPU::_andi() {
	D_rS; D_rA; D_UIMM;
	r.gpr[rA] = r.gpr[rS] & UIMM;
	r.setcr0(r.gpr[rA]);
}

void CPU::_addi() {
	D_rD; D_rA; D_SIMM;
	r.gpr[rD] = rA_0 + SIMM;
}

void CPU::_stwu() {
	D_rS; D_rA; D_d16;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");

	m.ww(r.gpr[rA] + d, r.gpr[rS]);
	r.gpr[rA] += d;
}

void CPU::_b() {
	D_LI; D_AA; D_LK;
	/*if(LK) {
	++function_level;
	DEGUB("\t//Entering level %i\n", function_level);
	}*/
	DWORD target;
	if(AA) {
		target = LI;
	} else {
		target = cia + LI;
	}
	SET_FETCH(target);
	if(target == cia && g::timing_mode == g::TM_EXACT_FAST) {
		cc.addCycles(1 >> BLOCK_SIZE_EXPONENT);
	}
	if(LK)
		r.lr = cia + 4;

	last_branch_unconditional = true;
}

void CPU::_addis() {
	D_rD; D_rA; D_SIMM;
	r.gpr[rD] = rA_0 + (int(SIMM) << 16);
}

void CPU::_ori() {
	D_rS; D_rA; D_UIMM;
	r.gpr[rA] = r.gpr[rS] | UIMM;
}

void CPU::_bclr() {
	D_BO; D_BI; D_LK;
	if(!getbitr(BO, 2))
		r.ctr--;
	bool ctr_ok = getbitr(BO, 2) || ((r.ctr != 0) != getbitr(BO, 1));
	bool cond_ok = getbitr(BO, 4) || (getbit(r.cr, BI) == getbitr(BO, 3));
	if(ctr_ok && cond_ok) {
		if(!LK && g::use_map) { //return; logging
			size_t from = get_symbol_index(cia);
			if(from != -1) {
				size_t to = get_symbol_index(r.lr);
				//DEGUB("blr from 0x%08X to 0x%08X. r3 = 0x%08X\n", cia, r.lr, r.gpr[3]);
				DEGUB("blr from ");
				degub_symbol(from, cia);
				DEGUB(" to ");
				if(to != -1) {
					degub_symbol(to, r.lr);
				} else {
					DEGUB("0x%08X", r.lr);
				}
				DEGUB(". r3 = 0x%08X\n", r.gpr[3]);
			}
		}
		SET_FETCH(r.lr & 0xFFFFFFFC);
		if(LK)
			r.lr = cia + 4;
	}
	last_branch_unconditional = BO == 0x14; //0x14 == "branch always"
}

void CPU::_dcbf() {
	D_rA; D_rB;

	DWORD EA = rA_0 + r.gpr[rB];
	if(g::cache_enabled && EA >= CACHE_BASE && EA < CACHE_BASE + CACHE_SIZE) {
		DEGUB("dcbf 0x%08X\n", EA);
		EA &= CACHE_BLOCK_MASK;
		//memcpy(m.getp_physical(PHYSICALIZE(EA), 32), m.getp_translated(EA, 32), 32);
		m.read_cached(EA, 32, m.getp_physical(PHYSICALIZE(EA), 32));
	}
}

void CPU::_mfmsr() {
	D_rD;
	r.gpr[rD] = r.getmsr();
	//DEGUB("mfmsr %08X\n", r.msr);
}

void CPU::_mtmsr() {
	D_rS;
	r.setmsr(r.gpr[rS]);
	if(g::lowmem_log) {
		DEGUB("mtmsr %08X", r.getmsr());
		for(int i=0; i<32; i++) {
			if(getbit(r.getmsr(), i)) switch(i) {
			case 13:
				DEGUB(" POW ignored");
				break;
			case 15:
				DEGUB(" ILE unemulated");
				break;
			case 16:
				DEGUB(" EE acknowledged");
				break;
			case 17:
				DEGUB(" PR unemulated");
				break;
			case 18:
				DEGUB(" FP unemulated");
				break;
			case 19:
				DEGUB(" ME ignored");
				break;
			case 20:
				DEGUB(" FE0 unemulated");
				break;
			case 21:
				DEGUB(" SE unemulated");
				break;
			case 22:
				DEGUB(" BE unemulated");
				break;
			case 23:
				DEGUB(" FE1 unemulated");
				break;
			case 25:
				DEGUB(" IP acknowledged");
				break;
			case 26:
				DEGUB(" IR unemulated");
				break;
			case 27:
				DEGUB(" DR unemulated");
				break;
			case 29:
				DEGUB(" PM unemulated");
				break;
			case 30:
				DEGUB(" RI unemulated");
				break;
			case 31:
				DEGUB(" LE unemulated");
				break;
			default:
				DEGUB(" Unknown MSR bit %i set", i);
			}
		}
		DEGUB("\n");
	}
}

void CPU::_mfspr() {
	D_rD; D_spr;
	if(!r.mfspr_is_valid(spr)) {
		DEGUB("Invalid mfSPR: %i\n", spr);
		throw interp_fatal_exception("illegal instruction type");
	}

	if(mfspr_requires_processing(spr))
		mfspr_with_processing(opcode);
	else
		r.gpr[rD] = *r.getmfspr(spr);
}

void CPU::_oris() {
	D_rS; D_rA; D_UIMM;
	r.gpr[rA] = r.gpr[rS] | (DWORD(UIMM) << 16);
}

void CPU::_or() {
	D_rS; D_rA; D_rB; D_Rc;
	r.gpr[rA] = r.gpr[rS] | r.gpr[rB];

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_mtspr() {
	D_rS; D_spr;
	if(!r.mtspr_is_valid(spr)) {
		DEGUB("Invalid mtSPR: %i\n", spr);
		throw interp_fatal_exception("illegal instruction type");
	}

	if(mtspr_requires_processing(spr))
		mtspr_with_processing(opcode);
	else
		*r.getmtspr(spr) = r.gpr[rS];
}

void CPU::_bc() {
	D_BO; D_BI; D_BD; D_AA; D_LK;
	if(!getbitr(BO, 2))
		r.ctr--;
	bool ctr_ok = getbitr(BO, 2) || ((r.ctr != 0) != getbitr(BO, 1));
	bool cond_ok = getbitr(BO, 4) || (getbit(r.cr, BI) == getbitr(BO, 3));
	if(ctr_ok && cond_ok) {
		/*if(LK) {
		++function_level;
		DEGUB("\t//Entering level %i\n", function_level);
		}*/
		if(AA) {
			SET_FETCH(BD);
		} else {
			SET_FETCH(cia + BD);
		}

		if(LK)
			r.lr = cia + 4;
	}
	last_branch_unconditional = BO == 0x14; //0x14 == "branch always"
}

void CPU::_lwz() {
	D_rD; D_rA; D_d16;
	r.gpr[rD] = m.rw(rA_0 + d);
}

void CPU::_sync() {
}

void CPU::_rlwinm() {
	D_rS; D_rA; D_SH; D_MB; D_ME; D_Rc;
	r.gpr[rA] = _rotl(r.gpr[rS], SH) & makemaskw(MB, ME);

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_cmpli() {
	D_crfD; D_L; D_rA; D_UIMM;
	if(L)
		throw interp_fatal_exception("Invalid instruction form");

	DWORD c;
	if(r.gpr[rA] < UIMM)
		c = CR_LT;
	else if(r.gpr[rA] > UIMM)
		c = CR_GT;
	else
		c = CR_EQ;

	r.setcr(crfD, c | (getflag(r.xer, XER_SO) ? 1 : 0));
}

void CPU::_isync() {
}

void CPU::_crxor() {
	D_crbD; D_crbA; D_crbB;
	setflags(r.cr, makeflag(crbD), getbit(r.cr, crbA) != getbit(r.cr, crbB));
}

void CPU::_add() {
	D_rD; D_rA; D_rB; D_OE; D_Rc;

	if(OE) {
		DWORD a = r.gpr[rA];
		DWORD b = r.gpr[rB];
		DWORD d;
		bool ov;
		MYASSERT(sizeof(bool) == 1);
		__asm {
			mov eax, a;
			add eax, b;
			mov d, eax;
			seto ov;
		}
		r.gpr[rD] = d;
		r.set_overflow(ov);
	} else {
		r.gpr[rD] = r.gpr[rA] + r.gpr[rB];
	}
	if(Rc)
		r.setcr0(r.gpr[rD]);
}

void CPU::_subf() {
	D_rD; D_rA; D_rB; D_OE; D_Rc;

	if(OE) {
		DWORD a = r.gpr[rA];
		DWORD b = r.gpr[rB];
		DWORD d;
		bool ov;
		MYASSERT(sizeof(bool) == 1);
		__asm {
			mov eax, b;
			sub eax, a;
			mov d, eax;
			seto ov;
		}
		r.gpr[rD] = d;
		r.set_overflow(ov);
	} else {
		r.gpr[rD] = r.gpr[rB] - r.gpr[rA];
	}
	if(Rc)
		r.setcr0(r.gpr[rD]);
}

void CPU::_cmp() {
	D_crfD; D_L; D_rA; D_rB;
	if(L)
		throw interp_fatal_exception("Invalid instruction form");

	DWORD c;
	if(int(r.gpr[rA]) < int(r.gpr[rB]))
		c = CR_LT;
	else if(int(r.gpr[rA]) > int(r.gpr[rB]))
		c = CR_GT;
	else
		c = CR_EQ;

	r.setcr(crfD, c | (getflag(r.xer, XER_SO) ? 1 : 0));
}

void CPU::_xori() {
	D_rS; D_rA; D_UIMM;
	r.gpr[rA] = r.gpr[rS] ^ UIMM;
}

void CPU::_cmpi() {
	D_crfD; D_L; D_rA; D_SIMM;
	if(L)
		throw interp_fatal_exception("Invalid instruction form");

	DWORD c;
	if(int(r.gpr[rA]) < SIMM)
		c = CR_LT;
	else if(int(r.gpr[rA]) > SIMM)
		c = CR_GT;
	else
		c = CR_EQ;

	r.setcr(crfD, c | (getflag(r.xer, XER_SO) ? 1 : 0));
}

void CPU::_lhz() {
	D_rD; D_rA; D_d16;
	r.gpr[rD] = m.rh(rA_0 + d);
}

void CPU::_stb() {
	D_rS; D_rA; D_d16;
	m.wb(rA_0 + d, (BYTE)r.gpr[rS]);
}

void CPU::_lbz() {
	D_rD; D_rA; D_d16;
	r.gpr[rD] = m.rb(rA_0 + d);
}

void CPU::_mulli() {
	D_rD; D_rA; D_SIMM;
	r.gpr[rD] = DWORD(__int64((int)r.gpr[rA]) * __int64(SIMM));
}

void CPU::_mulhw() {
	D_rD; D_rA; D_rB; D_Rc;
	r.gpr[rD] = DWORD((__int64((int)r.gpr[rA]) * __int64((int)r.gpr[rB])) >> 32);

	if(Rc)
		r.setcr0(r.gpr[rD]);
}

void CPU::_srawi() {
	D_rS; D_rA; D_SH; D_Rc;
	setflags(r.xer, XER_CA, signw(r.gpr[rS]) &&
		getbitsw(r.gpr[rS], 32-SH, 31) != 0 && SH != 0);

	DWORD d = r.gpr[rS] >> SH;
	if(getbit(r.gpr[rS], 0) && SH != 0)
		d |= makemaskw(0, SH);

	r.gpr[rA] = d;

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_stfd() {
	D_frS; D_rA; D_d16;
	double temp = FPR_PS_D(frS);
	m.wd(rA_0 + d, MAKE(QWORD, temp));
}

void CPU::_lfd() {
	D_frD; D_rA; D_d16;
	r.fpr[frD].dword = m.rd(rA_0 + d);
}

void CPU::_extsb() {
	D_rS; D_rA; D_Rc;
	r.gpr[rA] = (int)(signed char)r.gpr[rS];

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_mtfsb1() {
	D_crbD; D_Rc;
	if(crbD == 1 || crbD == 2)
		throw interp_fatal_exception("Invalid instruction (?)");

	setflags(r.fpscr, makeflag(crbD), true);
	r.update_fpscr_fex_vx();
	//r.update_fpscr_fx();  //Uncomment for proper emulation
	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_andc() {
	D_rS; D_rA; D_rB; D_Rc;
	r.gpr[rA] = r.gpr[rS] & ~r.gpr[rB];

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_cntlzw() {
	D_rS; D_rA; D_Rc;
	DWORD n=0;
	for(; n<32; n++)
		if(getbit(r.gpr[rS], n))
			break;

	r.gpr[rA] = n;

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_sth() {
	D_rS; D_rA; D_d16;
	m.wh(rA_0 + d, (WORD)r.gpr[rS]);
}

void CPU::_and() {
	D_rS; D_rA; D_rB; D_Rc;
	r.gpr[rA] = r.gpr[rS] & r.gpr[rB];

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_xor() {
	D_rS; D_rA; D_rB; D_Rc;
	r.gpr[rA] = r.gpr[rS] ^ r.gpr[rB];

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_neg() {
	D_rD; D_rA; D_OE; D_Rc;
	if(OE)
		r.set_overflow(r.gpr[rA] == 0x80000000);

	r.gpr[rD] = (~r.gpr[rA]) + 1;
	//r.gpr[rD] = -int(r.gpr[rA]);

	if(Rc)
		r.setcr0(r.gpr[rD]);
}

void CPU::_extsh() {
	D_rS; D_rA; D_Rc;
	r.gpr[rA] = (signed short)r.gpr[rS];

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_addze() {
	D_rD; D_rA; D_OE; D_Rc;

	DWORD a = r.gpr[rA];
	DWORD d;
	DWORD xer = r.xer;
	bool ca, ov;
	MYASSERT(sizeof(bool) == 1);
	__asm {
		bt xer, 31-2;  //xer[ca]
		mov eax, a;
		adc eax, 0;
		mov d, eax;
		setc ca;
		seto ov;
	}
	r.gpr[rD] = d;
	setflags(r.xer, XER_CA, ca);
	if(OE)
		r.set_overflow(ov);
	if(Rc)
		r.setcr0(d);
}

void CPU::_mtfsf() {
	D_FM; D_frB; D_Rc;
	DWORD bitmask=0;
	for(int i=0; i<8; i++) if(getbit(FM, 24+i)) {
		bitmask |= makemaskw(i*4, i*4+3);
	}
	setflags(r.fpscr, bitmask, false);
	setflags(r.fpscr, bitmask & r.fpr[frB].loword, true);
	r.update_fpscr_fex_vx();

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_lwzx() {
	D_rD; D_rA; D_rB;
	r.gpr[rD] = m.rw(rA_0 + r.gpr[rB]);
}

void CPU::_subfic() {
	D_rD; D_rA; D_SIMM;

	DWORD a = ~r.gpr[rA];
	DWORD d;
	bool ca;
	MYASSERT(sizeof(bool) == 1);
	__asm {
		stc;
		movsx eax, SIMM;
		adc eax, a;
		mov d, eax;
		setc ca;
	}
	r.gpr[rD] = d;
	setflags(r.xer, XER_CA, ca);
}

void CPU::_lfs() {
	D_frD; D_rA; D_d16;

	DWORD dword = m.rw(rA_0 + d);
	SET_FPR_SINGLE(MAKE(float, dword));
}

void CPU::_stwx() {
	D_rS; D_rA; D_rB;
	m.ww(rA_0 + r.gpr[rB], r.gpr[rS]);
}

void CPU::_xoris() {
	D_rS; D_rA; D_UIMM;
	r.gpr[rA] = r.gpr[rS] ^ (DWORD(UIMM) << 16);
}

void CPU::_fsub() {
	D_frD; D_frA; D_frB; D_Rc;

	SET_FPRD_FPRF(FPR_PS_D(frA) - FPR_PS_D(frB));

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_addic() {
	D_rD; D_rA; D_SIMM;
	setflags(r.xer, XER_CA, carry(r.gpr[rA], SIMM));
	r.gpr[rD] = r.gpr[rA] + SIMM;
}

void CPU::_subfe() {
	D_rD; D_rA; D_rB; D_OE; D_Rc;

	DWORD a = ~r.gpr[rA];
	DWORD b = r.gpr[rB];
	DWORD xer = r.xer;
	DWORD d;
	bool ca, ov;
	MYASSERT(sizeof(bool) == 1);
	__asm {
		bt xer, 31-2;  //xer[ca]
		mov eax, b;
		adc eax, a;
		mov d, eax;
		setc ca;
		seto ov;
	}
	r.gpr[rD] = d;
	setflags(r.xer, XER_CA, ca);
	if(OE)
		r.set_overflow(ov);
	if(Rc)
		r.setcr0(d);
}

void CPU::_lbzx() {
	D_rD; D_rA; D_rB;
	r.gpr[rD] = m.rb(rA_0 + r.gpr[rB]);
}

void CPU::_mullw() {
	D_rD; D_rA; D_rB; D_OE; D_Rc;

	if(OE) {
		DWORD a = r.gpr[rA];
		DWORD b = r.gpr[rB];
		DWORD d;
		bool ov;
		MYASSERT(sizeof(bool) == 1);
		__asm {
			mov eax, a;
			imul b;
			mov d, eax;
			seto ov;
		}
		r.gpr[rD] = d;
		r.set_overflow(ov);
	} else {
		r.gpr[rD] = r.gpr[rA] * r.gpr[rB];
	}
	if(Rc)
		r.setcr0(r.gpr[rD]);
}

void CPU::_addic_() {
	D_rD; D_rA; D_SIMM;
	setflags(r.xer, XER_CA, carry(r.gpr[rA], SIMM));
	r.gpr[rD] = r.gpr[rA] + SIMM;
	r.setcr0(r.gpr[rD]);
}

void CPU::_lha() {
	D_rD; D_rA; D_d16;
	r.gpr[rD] = (short)m.rh(rA_0 + d);
}

void CPU::_stbx() {
	D_rS; D_rA; D_rB;
	m.wb(rA_0 + r.gpr[rB], (BYTE)r.gpr[rS]);
}

void CPU::_sthx() {
	D_rS; D_rA; D_rB;
	m.wh(rA_0 + r.gpr[rB], (WORD)r.gpr[rS]);
}

void CPU::_bcctr() {
	D_BO; D_BI; D_LK;
	if(!getbitr(BO, 2))
		throw interp_fatal_exception("Invalid instruction form");

	bool cond_ok = getbitr(BO, 4) || (getbit(r.cr, BI) == getbitr(BO, 3));
	if(cond_ok) {
		/*if(LK) {
		++function_level;
		DEGUB("\t//Entering level %i\n", function_level);
		}*/
		SET_FETCH(r.ctr & 0xFFFFFFFC);
		if(LK)
			r.lr = cia + 4;
	}
	last_branch_unconditional = getbitr(BO, 4);
}

void CPU::_slw() {
	D_rS; D_rA; D_rB; D_Rc;
	r.gpr[rA] = getbit(r.gpr[rB], 26) ? 0 : r.gpr[rS] << (r.gpr[rB] & 0x0000001F);

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_sraw() {
	D_rS; D_rA; D_rB; D_Rc;

	DWORD SH = r.gpr[rB] & 0x0000001F;
	DWORD d=0;
	if(getbit(r.gpr[rB], 26)) {
		if(signw(r.gpr[rS]))
			d = makemaskw(0, 31);
		setflags(r.xer, XER_CA, signw(r.gpr[rS]));
	} else {
		d = r.gpr[rS] >> SH;
		if(signw(r.gpr[rS]) && SH != 0)
			d |= makemaskw(0, SH);
		setflags(r.xer, XER_CA, signw(r.gpr[rS]) &&
			getbitsw(r.gpr[rS], 32-SH, 31) != 0 && SH != 0);
	}

	r.gpr[rA] = d;

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_divw() {
	D_rD; D_rA; D_rB; D_OE; D_Rc;
	if((r.gpr[rA] == 0x80000000 && r.gpr[rB] == -1) || r.gpr[rB] == 0) {
		if(OE)
			r.set_overflow(true);
		//undefined, tested with GekkoTest
		if(r.gpr[rA] & 0x80000000)
			r.gpr[rD] = (DWORD)-1;
		else
			r.gpr[rD] = 0;
	} else {
		if(OE)
			r.set_overflow(false);
		r.gpr[rD] = int(r.gpr[rA]) / int(r.gpr[rB]);
	}

	if(Rc)
		r.setcr0(r.gpr[rD]);
}

void CPU::_adde() {
	D_rD; D_rA; D_rB; D_OE; D_Rc;

	DWORD a = r.gpr[rA];
	DWORD b = r.gpr[rB];
	DWORD d;
	DWORD xer = r.xer;
	bool ca, ov;
	MYASSERT(sizeof(bool) == 1);
	__asm {
		bt xer, 31-2;  //xer[ca]
		mov eax, a;
		adc eax, b;
		mov d, eax;
		setc ca;
		seto ov;
	}
	r.gpr[rD] = d;
	setflags(r.xer, XER_CA, ca);
	if(OE)
		r.set_overflow(ov);
	if(Rc)
		r.setcr0(d);
}

void CPU::_cmpl() {
	D_crfD; D_L; D_rA; D_rB;
	if(L)
		throw interp_fatal_exception("Invalid instruction form");

	DWORD c;
	if(r.gpr[rA] < r.gpr[rB])
		c = CR_LT;
	else if(r.gpr[rA] > r.gpr[rB])
		c = CR_GT;
	else
		c = CR_EQ;

	r.setcr(crfD, c | (getflag(r.xer, XER_SO) ? 1 : 0));
}

void CPU::_mcrf() {
	D_crfD; D_crfS;
	r.setcr(crfD, getbitsw(r.cr, crfS*4, crfS*4+3));
}

void CPU::_mfcr() {
	D_rD;
	r.gpr[rD] = r.cr;
}

void CPU::_mtcrf() {
	D_rS; D_CRM;
	DWORD mask=0;
	for(int i=0; i<8; i++) {
		if(getbit(CRM, 24+i))
			mask |= makemaskw(i*4, i*4+3); 
	}

	r.cr = mask & r.gpr[rS] | ~mask & r.cr;
}

void CPU::_stfs() {
	D_frS; D_rA; D_d16;

	float f = (float)GET_FPR_SINGLE(frS);
	m.ww(rA_0 + d, MAKE(DWORD, f));
}

void CPU::_fadds() {
	D_frD; D_frA; D_frB; D_Rc;

	SET_FPR_SINGLE_FPRF(FPR_PS_D(frA) + FPR_PS_D(frB));

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_fsubs() {
	D_frD; D_frA; D_frB; D_Rc;

	SET_FPR_SINGLE_FPRF(FPR_PS_D(frA) - FPR_PS_D(frB));

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_cror() {
	D_crbD; D_crbA; D_crbB;
	setflags(r.cr, makeflag(crbD), getbit(r.cr, crbA) || getbit(r.cr, crbB));
}

void CPU::_subfc() {
	D_rD; D_rA; D_rB; D_OE; D_Rc;

	DWORD a = ~r.gpr[rA];
	DWORD b = r.gpr[rB];
	DWORD d;
	bool ca, ov;
	MYASSERT(sizeof(bool) == 1);
	__asm {
		stc;
		mov eax, b;
		adc eax, a;
		mov d, eax;
		setc ca;
		seto ov;
	}
	r.gpr[rD] = d;
	setflags(r.xer, XER_CA, ca);
	if(OE)
		r.set_overflow(ov);
	if(Rc)
		r.setcr0(d);
}

void CPU::_lbzu() {
	D_rD; D_rA; D_d16;
	if(rA == 0 || rA == rD)
		throw interp_fatal_exception("Invalid instruction form");

	r.gpr[rD] = m.rb(r.gpr[rA] + d);
	r.gpr[rA] += d;
}

void CPU::_nor() {
	D_rS; D_rA; D_rB; D_Rc;
	r.gpr[rA] = ~(r.gpr[rS] | r.gpr[rB]);

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_lwzu() {
	D_rD; D_rA; D_d16;
	if(rA == 0 || rA == rD)
		throw interp_fatal_exception("Invalid instruction form");

	r.gpr[rD] = m.rw(r.gpr[rA] + d);
	r.gpr[rA] += d;
}

void CPU::_dcbst() {
	//D_rA; D_rB;
}

void CPU::_sc() {
	//throw interp_fatal_exception("unemulated instruction SC");
	//std_interrupt(INTERRUPT_SC, cia + 4);
	if(!g::nerf_sc)
		interrupt.raise(INTERRUPT_SC);
}

void CPU::_mftb() {
	D_rD; D_spr;
	if(!(spr == 268 || spr == 269))
		throw interp_fatal_exception("illegal instruction type");

	r.gpr[rD] = (spr == 268) ? getTBL() : getTBU();
}

void CPU::_addc() {
	D_rD; D_rA; D_rB; D_OE; D_Rc;

	DWORD a = r.gpr[rA];
	DWORD b = r.gpr[rB];
	DWORD d;
	bool ov, ca;
	MYASSERT(sizeof(bool) == 1);
	__asm {
		mov eax, a;
		add eax, b;
		mov d, eax;
		setc ca;
		seto ov;
	}
	r.gpr[rD] = d;
	setflags(r.xer, XER_CA, ca);
	if(OE)
		r.set_overflow(ov);
	if(Rc)
		r.setcr0(d);
}

void CPU::_stmw() {
	D_rS; D_rA; D_d16;
	DWORD EA = rA_0 + d;
	if((EA & 0x3) != 0)
		throw interp_fatal_exception("DSI Exception (0x00300)");

	while(rS <= 31) {
		m.ww(EA, r.gpr[rS]);
		rS++;
		EA += 4;
	}
}

void CPU::_icbi() {
	//D_rA; D_rB;
}

void CPU::_lmw() {
	D_rD; D_rA; D_d16;
	if(rA >= rD)
		throw interp_fatal_exception("Invalid instruction form");

	DWORD EA = rA_0 + d;
	if((EA & 0x3) != 0)
		throw interp_fatal_exception("DSI Exception (0x00300)");

	while(rD <= 31) {
		r.gpr[rD] = m.rw(EA);
		rD++;
		EA += 4;
	}
}

void CPU::_lhzx() {
	D_rD; D_rA; D_rB;
	r.gpr[rD] = m.rh(rA_0 + r.gpr[rB]);
}

void CPU::_mulhwu() {
	D_rD; D_rA; D_rB; D_Rc;
	r.gpr[rD] = DWORD((QWORD(r.gpr[rA]) * QWORD(r.gpr[rB])) >> 32);

	if(Rc)
		r.setcr0(r.gpr[rD]);
}

void CPU::_rlwimi() {
	D_rS; D_rA; D_SH; D_MB; D_ME; D_Rc;
	r.gpr[rA] = (r.gpr[rA] & ~makemaskw(MB, ME)) |
		(_rotl(r.gpr[rS], SH) & makemaskw(MB, ME));

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_dcbi() {
	//D_rA; D_rB;
}

void CPU::_stbu() {
	D_rS; D_rA; D_d16;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");

	m.wb(r.gpr[rA] + d, (BYTE)r.gpr[rS]);
	r.gpr[rA] += d;
}

void CPU::_nand() {
	D_rS; D_rA; D_rB; D_Rc;
	r.gpr[rA] = ~(r.gpr[rS] & r.gpr[rB]);

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_rfi() {
	/*if(g::log_interrupts) {
	DEGUB("RFI: ");
	}*/
	r.setmsr(((0x8780FF73 & r.srr1) | ((~0x8780FF73) & r.getmsr())) & (~0x00040000));
	SET_FETCH(r.srr0 & 0xFFFFFFFC);
}

void CPU::_divwu() {
	D_rD; D_rA; D_rB; D_OE; D_Rc;
	if(r.gpr[rB] == 0) {
		if(OE)
			r.set_overflow(true);
		r.gpr[rD] = 0;  //undefined, tested with GekkoTest
	} else {
		if(OE)
			r.set_overflow(false);
		r.gpr[rD] = r.gpr[rA] / r.gpr[rB];
	}

	if(Rc)
		r.setcr0(r.gpr[rD]);
}

void CPU::_srw() {
	D_rS; D_rA; D_rB; D_Rc;
	r.gpr[rA] = getbit(r.gpr[rB], 26) ? 0 : r.gpr[rS] >> (r.gpr[rB] & 0x0000001F);

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_stfsx() {
	D_frS; D_rA; D_rB;

	float f = (float)GET_FPR_SINGLE(frS);
	m.ww(rA_0 + r.gpr[rB], MAKE(DWORD, f));
}

void CPU::_lhax() {
	D_rD; D_rA; D_rB;
	r.gpr[rD] = (short)m.rh(rA_0 + r.gpr[rB]);
}

void CPU::_lfsx() {
	D_frD; D_rA; D_rB;

	DWORD dword = m.rw(rA_0 + r.gpr[rB]);
	SET_FPR_SINGLE(MAKE(float, dword));
}

//note: this may not change the sign of NaNs, as it should.
void CPU::_fneg() {
	D_frD; D_frB; D_Rc;

	SET_FPRD(-FPR_PS_D(frB));

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_fmul() {
	D_frD; D_frA; D_frC; D_Rc;

	SET_FPRD_FPRF(FPR_PS_D(frA) * FPR_PS_D(frC));

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_frsp() {
	D_frD; D_frB; D_Rc;

	r.set_fpscr_fprf(r.fpr[frD].d = (float)r.fpr[frB].d);

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_fdivs() {
	D_frD; D_frA; D_frB; D_Rc;

	SET_FPR_SINGLE_FPRF(FPR_PS_D(frA) / FPR_PS_D(frB));

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_fmadds() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;

	SET_FPR_SINGLE_FPRF(FPR_PS_D(frA) * FPR_PS_D(frC) + FPR_PS_D(frB));

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_fmuls() {
	D_frD; D_frA; D_frC; D_Rc;

	SET_FPR_SINGLE_FPRF(FPR_PS_D(frA) * FPR_PS_D(frC));

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_fmsubs() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;

	SET_FPR_SINGLE_FPRF(FPR_PS_D(frA) * FPR_PS_D(frC) - FPR_PS_D(frB));

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_fmr() {
	D_frD; D_frB; D_Rc;

	r.fpr[frD] = r.fpr[frB];

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_fcmpu() {
	D_crfD; D_frA; D_frB;
	DWORD c;

	double dA, dB;
	dA = FPR_PS_D(frA);
	dB = FPR_PS_D(frB);
	if(_isnan(dA) || _isnan(dB))
		c = 1;
	else if(dA < dB)
		c = 8;
	else if(dA > dB)
		c = 4;
	else
		c = 2;

	r.setcr(crfD, c);
	setbits(r.fpscr, 16, 19, c);
}

void CPU::_fctiwz() {
	D_frD; D_frB; D_Rc;

	r.fpr[frD].loword = (int)FPR_PS_D(frB);
	r.fpr[frD].hiword = UNDEFINED_PPCWORD;

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_fadd() {
	D_frD; D_frA; D_frB; D_Rc;

	SET_FPRD_FPRF(FPR_PS_D(frA) + FPR_PS_D(frB));

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_fmadd() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;

	SET_FPRD_FPRF(FPR_PS_D(frA) * FPR_PS_D(frC) + FPR_PS_D(frB));

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_lfdx() {
	D_frD; D_rA; D_rB;
	r.fpr[frD].dword = m.rd(rA_0 + r.gpr[rB]);
}

void CPU::_addme() {
	D_rD; D_rA; D_OE; D_Rc;

	DWORD a = r.gpr[rA];
	DWORD d;
	DWORD xer = r.xer;
	bool ca, ov;
	MYASSERT(sizeof(bool) == 1);
	__asm {
		bt xer, 31-2;  //xer[ca]
		mov eax, a;
		adc eax, -1;
		mov d, eax;
		setc ca;
		seto ov;
	}
	r.gpr[rD] = d;
	setflags(r.xer, XER_CA, ca);
	if(OE)
		r.set_overflow(ov);
	if(Rc)
		r.setcr0(d);
}

void CPU::_crnor() {
	D_crbD; D_crbA; D_crbB;
	setflags(r.cr, makeflag(crbD), !(getbit(r.cr, crbA) || getbit(r.cr, crbB)));
}

void CPU::_fdiv() {
	D_frD; D_frA; D_frB; D_Rc;

	SET_FPRD_FPRF(FPR_PS_D(frA) / FPR_PS_D(frB));

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_andis() {
	D_rS; D_rA; D_UIMM;
	r.gpr[rA] = r.gpr[rS] & (DWORD(UIMM) << 16);
	r.setcr0(r.gpr[rA]);
}

void CPU::_twi() {
	D_TO; D_rA; D_SIMM;

	if(TO == 0)
		throw interp_fatal_exception("Unemulated instruction form");

	if((int(r.gpr[rA]) < SIMM && (TO & 0x10)) ||
		(int(r.gpr[rA]) > SIMM && (TO & 0x08)) ||
		(int(r.gpr[rA]) == SIMM && (TO & 0x04)) ||
		(r.gpr[rA] < DWORD(SIMM) && (TO & 0x02)) ||
		(r.gpr[rA] > DWORD(SIMM) && (TO & 0x01)))
	{
		r.srr1 = makeflag(14);
		SET_FETCH(cia);
		interrupt.raise(INTERRUPT_PROGRAM);
	}
}

void CPU::_fmsub() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;

	SET_FPRD_FPRF(FPR_PS_D(frA) * FPR_PS_D(frC) - FPR_PS_D(frB));

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_subfze() {
	D_rD; D_rA; D_OE; D_Rc;

	DWORD a = ~r.gpr[rA];
	DWORD xer = r.xer;
	DWORD d;
	bool ca, ov;
	MYASSERT(sizeof(bool) == 1);
	__asm {
		bt xer, 31-2;  //xer[ca]
		mov eax, a;
		adc eax, 0;
		mov d, eax;
		setc ca;
		seto ov;
	}
	r.gpr[rD] = d;
	setflags(r.xer, XER_CA, ca);
	if(OE)
		r.set_overflow(ov);
	if(Rc)
		r.setcr0(d);
}

void CPU::_subfme() {
	D_rD; D_rA; D_OE; D_Rc;

	DWORD a = ~r.gpr[rA];
	DWORD xer = r.xer;
	DWORD d;
	bool ca, ov;
	MYASSERT(sizeof(bool) == 1);
	__asm {
		bt xer, 31-2;  //xer[ca]
		mov eax, a;
		adc eax, -1;
		mov d, eax;
		setc ca;
		seto ov;
	}
	r.gpr[rD] = d;
	setflags(r.xer, XER_CA, ca);
	if(OE)
		r.set_overflow(ov);
	if(Rc)
		r.setcr0(d);
}

void CPU::_orc() {
	D_rS; D_rA; D_rB; D_Rc;
	r.gpr[rA] = r.gpr[rS] | ~r.gpr[rB];

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_lhzu() {
	D_rD; D_rA; D_d16;
	if(rA == 0 || rA == rD)
		throw interp_fatal_exception("Invalid instruction form");

	r.gpr[rD] = m.rh(r.gpr[rA] + d);
	r.gpr[rA] += d;
}

void CPU::_lfdu() {
	D_frD; D_rA; D_d16;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");

	r.fpr[frD].dword = m.rd(rA_0 + d);

	r.gpr[rA] += d;
}

void CPU::_dcbz() {
	//D_rA; D_rB;
}

void CPU::_lbzux() {
	D_rD; D_rA; D_rB;
	if(rA == 0 || rA == rD)
		throw interp_fatal_exception("Invalid instruction form");
	DWORD temp = r.gpr[rB];
	r.gpr[rD] = m.rb(r.gpr[rA] + r.gpr[rB]);
	//r.gpr[rA] += r.gpr[rB]; //what if rB == rD?
	r.gpr[rA] += temp;
}

void CPU::_psq_lu() {
	D_frD; D_rA; D_W; D_I; D_d20;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");

	if(!W)
		quantized_load_W0(frD, rA_0 + d, I);
	else
		quantized_load_W1(frD, rA_0 + d, I);
	r.gpr[rA] += d;
}

void CPU::_sthu() {
	D_rS; D_rA; D_d16;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");

	m.wh(r.gpr[rA] + d, (WORD)r.gpr[rS]);
	r.gpr[rA] += d;
}

void CPU::_fnmadd() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;

	SET_FPRD_FPRF(-(FPR_PS_D(frA) * FPR_PS_D(frC) + FPR_PS_D(frB)));

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_creqv() {
	D_crbD; D_crbA; D_crbB;
	setflags(r.cr, makeflag(crbD), getbit(r.cr, crbA) == getbit(r.cr, crbB));
}

//May not work on NaNs
void CPU::_fabs() {
	D_frD; D_frB; D_Rc;

	//_control87(_PC_53, _MCW_PC);  //useless
	//SET_FPRD(fabs(FPR_PS_D(frB)));  //bungles NaNs
	r.fpr[frD].hiword = r.fpr[frB].hiword & 0x7FFFFFFF;
	r.fpr[frD].loword = r.fpr[frB].loword;

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_mtsrin() {
	D_rS; D_rB;

	r.sr[getbitsw(r.gpr[rB], 0, 3)] = r.gpr[rS];
	DEGUB("mtSR%i 0x%08X\n", getbitsw(r.gpr[rB], 0, 3),
		r.sr[getbitsw(r.gpr[rB], 0, 3)]);
}

void CPU::_mfsr() {
	D_rD; D_SR;
	r.gpr[rD] = r.sr[SR];
}

void CPU::_mffs() {
	D_frD; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction form");

	r.fpr[frD].loword = r.fpscr;
	r.fpr[frD].hiword = UNDEFINED_PPCWORD;
}

void CPU::_tlbie() {
	//D_rB;
}

void CPU::_stswi() {
	D_rS; D_rA; D_NB;

	DWORD n = 0;
	DWORD EA = rA_0;
	while(n < ((NB == 0) ? 32 : NB)) {
		m.wb(EA, *(((BYTE*)(&r.gpr[rS])) + (3 - (n % 4))));
		EA++;
		n++;
		if(n % 4 == 0)
			rS = (rS + 1) % 32;
	}
}

void CPU::_lwarx() {
	D_rD; D_rA; D_rB;

	DWORD ea = rA_0 + r.gpr[rB];
	if(ea % 4 != 0)
		throw interp_fatal_exception("Unaligned lwarx");

	reserve_address = PHYSICALIZE(g::advanced_mode ? m.translate_d_read(ea) : ea);
	reserve_data = r.gpr[rD] = m.rw(ea);
	reserve_bit = true;
	if(m.isDegubOn()) {
		DEGUB("rwar[%08X", ea);
		if(g::advanced_mode) {
			DEGUB("->%08X", reserve_address);
		}
		DEGUB("]%08X\n", r.gpr[rD]);
	}
}

void CPU::_stwcx_() {
	D_rS; D_rA; D_rB;
	DWORD ea = rA_0 + r.gpr[rB];

	if(ea % 4 != 0)
		throw interp_fatal_exception("Unaligned stwcx.");

	setflags(r.cr, CR0_LT | CR0_GT | CR0_EQ | CR0_SO, false);
	if(reserve_bit) {
		reserve_bit = false;
		DWORD pa;
		try {
			pa = PHYSICALIZE(g::advanced_mode ? m.translate_d_write(ea) : ea);
		} catch(exception &e) {
			DEGUB("exception in stwcx. @ %08X while trying to access %08X\n", cia, ea);
			throw interp_fatal_exception(e.what());
		}
		//the hrw might cause problems with hardware registers that change on read
		if((pa == reserve_address && reserve_data == m.rw(ea)) || pa != reserve_address) {
			//we may not throw a page fault until/unless we get here,
			//'cause that's how the Gekko works.
			m.ww(ea, r.gpr[rS]);
			setflags(r.cr, CR0_EQ, true);
			if(m.isDegubOn()) {
				DEGUB("wwar[%08X", ea);
				if(g::advanced_mode) {
					DEGUB("->%08X", pa);
				}
				DEGUB("]%08X\n", r.gpr[rS]);
			}
		} else {
			DEGUB("pa=%08X, ra=%08X. stwcx. store @ %08X to %08X failed!\n",
				pa, reserve_address, cia, ea);
		}
	}
	setflags(r.cr, CR0_SO, getflag(r.xer, XER_SO));
}

void CPU::_lwzux() {
	D_rD; D_rA; D_rB;
	if(rA == 0 || rA == rD)
		throw interp_fatal_exception("Invalid instruction form");
	DWORD temp = r.gpr[rB];
	r.gpr[rD] = m.rw(rA_0 + r.gpr[rB]);
	//r.gpr[rA] += r.gpr[rB]; //what if rB == rD?
	r.gpr[rA] += temp;
}

void CPU::_dcbt() {
	//D_rA; D_rB;
}

void CPU::_lswi() {
	D_rD; D_rA; D_NB;

	DWORD n = 0;
	DWORD EA = rA_0;
	while(n < ((NB == 0) ? 32 : NB)) {
		*(((BYTE*)(&r.gpr[rD])) + (3 - (n % 4))) = m.rb(EA);
		EA++;
		n++;
		if(n % 4 == 0)
			rD = (rD + 1) % 32;
	}
	//n is now equal to ((NB == 0) ? 32 : NB)
	if(n % 4 != 0)
		r.gpr[rD] &= makemaskw(0, (n % 4) * 8 - 1);
}

void CPU::_dcbtst() {
	//D_rA; D_rB;
}

void CPU::_rlwnm() {
	D_rS; D_rA; D_rB; D_MB; D_ME; D_Rc;
	r.gpr[rA] = _rotl(r.gpr[rS], getbitsw(r.gpr[rB], 27, 31)) & makemaskw(MB, ME);

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_stwux() {
	D_rS; D_rA; D_rB;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");

	m.ww(r.gpr[rA] + r.gpr[rB], r.gpr[rS]);
	r.gpr[rA] += r.gpr[rB];
}

void CPU::_stwbrx() {
	D_rS; D_rA; D_rB;
	m.ww(rA_0 + r.gpr[rB], swapw(r.gpr[rS]));
}

void CPU::_mfsrin() {
	D_rD; D_rB;

	r.gpr[rD] = r.sr[getbitsw(r.gpr[rB], 0, 3)];
}

void CPU::_fsel() {
	D_frD; D_frA; D_frB; D_frC; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRD(FPR_PS_D((FPR_PS_D(frA) >= 0.0) ? frC : frB));
}

void CPU::_psq_l() {
	D_frD; D_rA; D_W; D_I; D_d20;
	if(!W)
		quantized_load_W0(frD, rA_0 + d, I);
	else
		quantized_load_W1(frD, rA_0 + d, I);
}

void CPU::_psq_st() {
	D_frS; D_rA; D_W; D_I; D_d20;
	if(!W)
		quantized_store_W0(rA_0 + d, I, frS);
	else
		quantized_store_W1(rA_0 + d, I, frS);
}

void CPU::_fcmpo() {
	D_crfD; D_frA; D_frB;
	DWORD c;

	double dA, dB;
	dA = FPR_PS_D(frA);
	dB = FPR_PS_D(frB);
	if(_isnan(dA) || _isnan(dB))
		c = 1;
	else if(dA < dB)
		c = 8;
	else if(dA > dB)
		c = 4;
	else
		c = 2;

	r.setcr(crfD, c);
	setbits(r.fpscr, 16, 19, c);
}

void CPU::_ps_merge01() {
	D_frD; D_frA; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRPS(PS0(frA), PS1(frB));
}

void CPU::_ps_merge10() {
	D_frD; D_frA; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	if(frD == frB) {
		double temp = PS0(frB);
		SET_FPRPS(PS1(frA), temp);
	} else {
		SET_FPRPS(PS1(frA), PS0(frB));
	}
}

void CPU::_ps_mr() {
	D_frD; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	PS0(frD) = PS0(frB);
	PS1(frD) = PS1(frB);
}

void CPU::_stfsu() {
	D_frS; D_rA; D_d16;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");

	float f = (float)GET_FPR_SINGLE(frS);
	m.ww(r.gpr[rA] + d, MAKE(DWORD, f));
	r.gpr[rA] += d;
}

void CPU::_stfdx() {
	D_frS; D_rA; D_rB;
	double temp = FPR_PS_D(frS);
	m.wd(rA_0 + r.gpr[rB], MAKE(QWORD, temp));
}

void CPU::_stbux() {
	D_rS; D_rA; D_rB;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");

	m.wb(r.gpr[rA] + r.gpr[rB], (BYTE)r.gpr[rS]);
	r.gpr[rA] += r.gpr[rB];
}

void CPU::_ps_mul() {
	D_frD; D_frA; D_frC; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRPS_FPRF(PS0(frA) * PS0(frC), PS1(frA) * PS1(frC));
}

void CPU::_ps_madd() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRPS_FPRF(PS0(frA) * PS0(frC) + PS0(frB),
		PS1(frA) * PS1(frC) + PS1(frB));
}

void CPU::_ps_sum0() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRPS_FPRF(PS0(frA) + PS1(frB), PS1(frC));
}

void CPU::_frsqrte() {
	D_frD; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRD_FPRF(1.0 / sqrt(FPR_PS_D(frB)));
}

void CPU::_fnmsubs() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPR_SINGLE_FPRF(-(FPR_PS_D(frA) * FPR_PS_D(frC) - FPR_PS_D(frB)));
}

void CPU::__wc_debug() {
	this->wc_debug();
}

void CPU::_ps_muls0() {
	D_frD; D_frA; D_frC; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	if(frD == frC) {
		double temp = PS1(frA) * PS0(frC);
		SET_FPRPS(PS0(frA) * PS0(frC), temp);
	} else {
		SET_FPRPS_FPRF(PS0(frA) * PS0(frC), PS1(frA) * PS0(frC));
	}
}

void CPU::_ps_muls1() {
	D_frD; D_frA; D_frC; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRPS_FPRF(PS0(frA) * PS1(frC), PS1(frA) * PS1(frC));
}

void CPU::_ps_msub() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRPS_FPRF(PS0(frA) * PS0(frC) - PS0(frB),
		PS1(frA) * PS1(frC) - PS1(frB));
}

//note: this may not change the sign of NaNs, as it should.
void CPU::_ps_neg() {
	D_frD; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRPS(-PS0(frB), -PS1(frB));
}

void CPU::_lfsu() {
	D_frD; D_rA; D_d16;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");

	DWORD dword = m.rw(rA_0 + d);
	SET_FPR_SINGLE(MAKE(float, dword));

	r.gpr[rA] += d;
}

void CPU::_eieio() {
}

void CPU::_ps_merge00() {
	D_frD; D_frA; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	if(frD == frB) {
		double temp = PS0(frB);
		SET_FPRPS(PS0(frA), temp);
	} else {
		SET_FPRPS(PS0(frA), PS0(frB));
	}
}

void CPU::_ps_merge11() {
	D_frD; D_frA; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRPS(PS1(frA), PS1(frB));
}

void CPU::_fnmadds() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPR_SINGLE_FPRF(-(FPR_PS_D(frA) * FPR_PS_D(frC) + FPR_PS_D(frB)));
}

void CPU::_ps_madds0() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");
	if(frD == frC) {
		double temp = PS1(frA) * PS0(frC) + PS1(frB);
		SET_FPRPS_FPRF(PS0(frA) * PS0(frC) + PS0(frB), temp);
	} else {
		SET_FPRPS_FPRF(PS0(frA) * PS0(frC) + PS0(frB),
			PS1(frA) * PS0(frC) + PS1(frB));
	}
}

void CPU::_ps_madds1() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRPS_FPRF(PS0(frA) * PS1(frC) + PS0(frB),
		PS1(frA) * PS1(frC) + PS1(frB));
}

void CPU::_ps_add() {
	D_frD; D_frA; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRPS_FPRF(PS0(frA) + PS0(frB), PS1(frA) + PS1(frB));
}

void CPU::_ps_sub() {
	D_frD; D_frA; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRPS_FPRF(PS0(frA) - PS0(frB), PS1(frA) - PS1(frB));
}

void CPU::_ps_cmpo0() {
	D_crfD; D_frA; D_frB;
	DWORD c;

	double dA = PS0(frA);
	double dB = PS0(frB);
	if(_isnan(dA) || _isnan(dB))
		c = 1;
	else if(dA < dB)
		c = 8;
	else if(dA > dB)
		c = 4;
	else
		c = 2;

	r.setcr(crfD, c);
	setbits(r.fpscr, 16, 19, c);
}

void CPU::_ps_cmpo1() {
	D_crfD; D_frA; D_frB;
	DWORD c;

	double dA = PS1(frA);
	double dB = PS1(frB);
	if(_isnan(dA) || _isnan(dB))
		c = 1;
	else if(dA < dB)
		c = 8;
	else if(dA > dB)
		c = 4;
	else
		c = 2;

	r.setcr(crfD, c);
	setbits(r.fpscr, 16, 19, c);
}

void CPU::_ps_cmpu0() {
	D_crfD; D_frA; D_frB;
	DWORD c;

	double dA = PS0(frA);
	double dB = PS0(frB);
	if(_isnan(dA) || _isnan(dB))
		c = 1;
	else if(dA < dB)
		c = 8;
	else if(dA > dB)
		c = 4;
	else
		c = 2;

	r.setcr(crfD, c);
	setbits(r.fpscr, 16, 19, c);
}

void CPU::_ps_cmpu1() {
	D_crfD; D_frA; D_frB;
	DWORD c;

	double dA = PS1(frA);
	double dB = PS1(frB);
	if(_isnan(dA) || _isnan(dB))
		c = 1;
	else if(dA < dB)
		c = 8;
	else if(dA > dB)
		c = 4;
	else
		c = 2;

	r.setcr(crfD, c);
	setbits(r.fpscr, 16, 19, c);
}

void CPU::_fres() {
	D_frD; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPR_SINGLE_FPRF(1.0 / FPR_PS_D(frB));
}

void CPU::_ps_nmadd() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRPS_FPRF(-(PS0(frA) * PS0(frC) + PS0(frB)),
		-(PS1(frA) * PS1(frC) + PS1(frB)));
}

void CPU::_ps_nmsub() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRPS_FPRF(-(PS0(frA) * PS0(frC) - PS0(frB)),
		-(PS1(frA) * PS1(frC) - PS1(frB)));
}

void CPU::_ps_sum1() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	double temp = PS0(frA) + PS1(frB);
	PS0(frD) = PS0(frC);
	r.set_fpscr_fprf(PS1(frD) = temp);	//NOTE!
}

void CPU::_eqv() {
	D_rS; D_rA; D_rB; D_Rc;
	r.gpr[rA] = ~(r.gpr[rS] ^ r.gpr[rB]);

	if(Rc)
		r.setcr0(r.gpr[rA]);
}

void CPU::_lhau() {
	D_rD; D_rA; D_d16;
	if(rA == 0 || rA == rD)
		throw interp_fatal_exception("Invalid instruction form");

	r.gpr[rD] = (short)m.rh(rA_0 + d);
	r.gpr[rA] += d;
}

void CPU::_lhaux() {
	D_rD; D_rA; D_rB;
	if(rA == 0 || rA == rD)
		throw interp_fatal_exception("Invalid instruction form");

	DWORD temp = r.gpr[rB];
	r.gpr[rD] = (short)m.rh(rA_0 + r.gpr[rB]);
	//r.gpr[rA] += r.gpr[rB]; //what if rB == rD?
	r.gpr[rA] += temp;
}

void CPU::_fnmsub() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;

	SET_FPRD_FPRF(-(FPR_PS_D(frA) * FPR_PS_D(frC) - FPR_PS_D(frB)));

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_psq_stu() {
	D_frS; D_rA; D_W; D_I; D_d20;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");

	if(!W)
		quantized_store_W0(r.gpr[rA] + d, I, frS);
	else
		quantized_store_W1(r.gpr[rA] + d, I, frS);
	r.gpr[rA] += d;
}

void CPU::_ps_rsqrte() {
	D_frD; D_frB; D_Rc;

	SET_FPRPS_FPRF(1.0 / sqrt(PS0(frB)), 1.0 / sqrt(PS1(frB)));  

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_ps_res() {
	D_frD; D_frB; D_Rc;

	SET_FPRPS_FPRF(1.0 / PS0(frB), 1.0 / PS1(frB));  

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::__oovpa() {
	DWORD i = getbitsw(opcode, 6, 31);
	DWORD ria = generic_oovpa_func(cia, i);
	SET_FETCH(ria);
}

void CPU::__gcm_dol_loaded() {
	this->do_oovpa();
}

void CPU::_mcrxr() {
	D_crfD;
	r.setcr(crfD, getbitsw(r.xer, 0, 3));
	r.xer &= 0x0FFFFFFF;
}

void CPU::_stfiwx() {
	D_frS; D_rA; D_rB;
	m.ww(rA_0 + r.gpr[rB], r.fpr[frS].loword);
}

void CPU::_fctiw() {
	D_frD; D_frB; D_Rc;

	r.fpr[frD].loword = (int)FPR_PS_D(frB);
	r.fpr[frD].hiword = UNDEFINED_PPCWORD;

	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_sthbrx() {
	D_rS; D_rA; D_rB;
	m.wh(rA_0 + r.gpr[rB], swaph((WORD)r.gpr[rS]));
}

void CPU::__osreport() {
	this->osreport();
}

void CPU::_stfdu() {
	D_frS; D_rA; D_d16;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");

	m.wd(r.gpr[rA] + d, r.fpr[frS].dword);
	r.gpr[rA] += d;
}

void CPU::_stfdux() {
	D_frS; D_rA; D_rB;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");

	m.wd(r.gpr[rA] + r.gpr[rB], r.fpr[frS].dword);
	r.gpr[rA] += r.gpr[rB];
}

void CPU::_mtfsb0() {
	D_crbD; D_Rc;
	if(crbD == 1 || crbD == 2)
		throw interp_fatal_exception("Invalid instruction (?)");

	setflags(r.fpscr, makeflag(crbD), false);
	r.update_fpscr_fex_vx();
	//r.update_fpscr_fx();  //Uncomment for proper emulation
	if(Rc)
		r.setcr(1, getbitsw(r.fpscr, 0, 3));
}

void CPU::_dcbz_l() {
	//D_rA; D_rB;
}

void CPU::_lwbrx() {
	D_rD; D_rA; D_rB;
	r.gpr[rD] = swapw(m.rw(rA_0 + r.gpr[rB]));
}

void CPU::_lhbrx() {
	D_rD; D_rA; D_rB;
	r.gpr[rD] = swaph(m.rh(rA_0 + r.gpr[rB]));
}

void CPU::_lfdux() {
	D_frD; D_rA; D_rB;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");
	r.fpr[frD].dword = m.rd(r.gpr[rA] + r.gpr[rB]);
	r.gpr[rA] += r.gpr[rB];
}

void CPU::_lfsux() {
	D_frD; D_rA; D_rB;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");
	DWORD dword = m.rw(r.gpr[rA] + r.gpr[rB]);
	SET_FPR_SINGLE(MAKE(float, dword));
	r.gpr[rA] += r.gpr[rB];
}

void CPU::_lhzux() {
	D_rD; D_rA; D_rB;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");
	DWORD temp = r.gpr[rB];
	r.gpr[rD] = m.rh(r.gpr[rA] + r.gpr[rB]);
	//r.gpr[rA] += r.gpr[rB]; //what if rB == rD?
	r.gpr[rA] += temp;
}

void CPU::_psq_lx() {
	D_frD; D_rA; D_rB; D_W; D_I;
	if(!W)
		quantized_load_W0(frD, rA_0 + r.gpr[rB], I);
	else
		quantized_load_W1(frD, rA_0 + r.gpr[rB], I);
}

void CPU::_psq_lux() {
	D_frD; D_rA; D_rB; D_W; D_I;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");
	if(!W)
		quantized_load_W0(frD, r.gpr[rA] + r.gpr[rB], I);
	else
		quantized_load_W1(frD, r.gpr[rA] + r.gpr[rB], I);
	r.gpr[rA] += r.gpr[rB];
}

void CPU::_psq_stx() {
	D_frS; D_rA; D_rB; D_W; D_I;
	if(!W)
		quantized_store_W0(rA_0 + r.gpr[rB], I, frS);
	else
		quantized_store_W1(rA_0 + r.gpr[rB], I, frS);
}

void CPU::_psq_stux() {
	D_frS; D_rA; D_rB; D_W; D_I;
	if(rA == 0)
		throw interp_fatal_exception("Invalid instruction form");
	if(!W)
		quantized_store_W0(r.gpr[rA] + r.gpr[rB], I, frS);
	else
		quantized_store_W1(r.gpr[rA] + r.gpr[rB], I, frS);
	r.gpr[rA] += r.gpr[rB];
}

void CPU::_mtsr() {
	D_SR; D_rS;
	r.sr[SR] = r.gpr[rS];
}

void CPU::_ps_abs() {
	D_frD; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	PS0(frD) = fabs(PS0(frB));
	PS1(frD) = fabs(PS1(frB));
}

void CPU::_ps_div() {
	D_frD; D_frA; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRPS_FPRF(PS0(frA) / PS0(frB), PS1(frA) / PS1(frB));
}

void CPU::_ps_nabs() {
	D_frD; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	PS0(frD) = -fabs(PS0(frB));
	PS1(frD) = -fabs(PS1(frB));
}

void CPU::_ps_sel() {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	if(Rc)
		throw interp_fatal_exception("Unemulated instruction mode");

	SET_FPRPS_FPRF((PS0(frA) >= 0.0) ? PS0(frC) : PS0(frB),
		(PS1(frA) >= 0.0) ? PS1(frC) : PS1(frB));
}

void CPU::_tlbsync() {
}

void CPU::_mcrfs() {
	D_crfD; D_crfS;
	r.setcr(crfD, getbitsw(r.fpscr, crfS*4, crfS*4+3));
}

void CPU::_crand() {
	D_crbD; D_crbA; D_crbB;
	setflags(r.cr, makeflag(crbD), getbit(r.cr, crbA) && getbit(r.cr, crbB));
}

void CPU::_crandc() {
	D_crbD; D_crbA; D_crbB;
	setflags(r.cr, makeflag(crbD), getbit(r.cr, crbA) && !getbit(r.cr, crbB));
}

void CPU::_crnand() {
	D_crbD; D_crbA; D_crbB;
	setflags(r.cr, makeflag(crbD), !(getbit(r.cr, crbA) && getbit(r.cr, crbB)));
}

void CPU::_crorc() {
	D_crbD; D_crbA; D_crbB;
	setflags(r.cr, makeflag(crbD), getbit(r.cr, crbA) || !getbit(r.cr, crbB));
}

void CPU::__dsp_quick_execute() {
	this->dsp_quick_execute();
}
